// app.js

// 全局变量
let students = [];
let selectedStudents = [];
let isAdmin = false;
let longPressTimer = null;

// 初始化函数
document.addEventListener('DOMContentLoaded', function() {
    loadStudents();
    setupEventListeners();
});

// 从后端加载学生数据
async function loadStudents() {
    try {
        const response = await fetch('/api/students');
        students = await response.json();
        renderSeats();
    } catch (error) {
        console.error('加载学生数据失败:', error);
        // 使用模拟数据
        students = generateMockData();
        renderSeats();
    }
}

// 生成模拟数据
function generateMockData() {
    const mockStudents = [];
    const names = ['张三', '李四', '王五', '赵六', '钱七', '孙八', '周九', '吴十'];
    
    for (let i = 0; i < 24; i++) {
        mockStudents.push({
            id: i + 1,
            name: names[i % names.length] + (Math.floor(i / names.length) + 1),
            score: Math.floor(Math.random() * 100),
            hasRecord: Math.random() > 0.7,
            recordDetails: Math.random() > 0.7 ? '迟到3次' : '',
            seat: i + 1
        });
    }
    
    return mockStudents;
}

// 渲染座位表
function renderSeats() {
    const seatGrid = document.getElementById('seatGrid');
    seatGrid.innerHTML = '';
    
    students.forEach(student => {
        const seatElement = document.createElement('div');
        seatElement.className = 'seat';
        if (student.hasRecord) {
            seatElement.classList.add('has-record');
        }
        seatElement.dataset.id = student.id;
        
        seatElement.innerHTML = `
            <div class="student-info">
                <div class="student-name">${student.name}</div>
                <div class="student-score">${student.score}分</div>
                ${student.hasRecord ? '<div class="student-record">有记过</div>' : ''}
            </div>
        `;
        
        seatGrid.appendChild(seatElement);
    });
    
    // 添加座位点击事件
    document.querySelectorAll('.seat').forEach(seat => {
        seat.addEventListener('click', handleSeatClick);
        seat.addEventListener('mousedown', startLongPress);
        seat.addEventListener('touchstart', startLongPress);
        seat.addEventListener('mouseup', endLongPress);
        seat.addEventListener('mouseleave', endLongPress);
        seat.addEventListener('touchend', endLongPress);
    });
}

// 设置事件监听器
function setupEventListeners() {
    // 登录按钮
    document.getElementById('loginBtn').addEventListener('click', showLoginModal);
    
    // 登录表单
    document.getElementById('loginForm').addEventListener('submit', handleLogin);
    
    // 单个学生操作
    document.getElementById('addPointsSingle').addEventListener('click', addPointsSingle);
    document.getElementById('subtractPointsSingle').addEventListener('click', subtractPointsSingle);
    document.getElementById('addRecordSingle').addEventListener('click', addRecordSingle);
    document.getElementById('removeRecordSingle').addEventListener('click', removeRecordSingle);
    
    // 批量操作
    document.getElementById('addPointsBtn').addEventListener('click', showBatchModal);
    document.getElementById('subtractPointsBtn').addEventListener('click', showBatchModal);
    document.getElementById('addRecordBtn').addEventListener('click', showBatchModal);
    document.getElementById('clearSelectionBtn').addEventListener('click', clearSelection);
    
    // 批量操作表单
    document.getElementById('batchAddPoints').addEventListener('click', batchAddPoints);
    document.getElementById('batchSubtractPoints').addEventListener('click', batchSubtractPoints);
    document.getElementById('batchAddRecord').addEventListener('click', batchAddRecord);
}

// 处理座位点击
function handleSeatClick(event) {
    const seat = event.currentTarget;
    const studentId = parseInt(seat.dataset.id);
    
    // 如果处于多选模式
    if (isAdmin && selectedStudents.length > 0) {
        toggleStudentSelection(studentId, seat);
        return;
    }
    
    // 否则显示学生信息
    showStudentInfo(studentId);
}

// 开始长按
function startLongPress(event) {
    if (!isAdmin) return;
    
    const seat = event.currentTarget;
    longPressTimer = setTimeout(() => {
        seat.classList.add('long-press-active');
        const studentId = parseInt(seat.dataset.id);
        toggleStudentSelection(studentId, seat);
        longPressTimer = null;
    }, 500);
}

// 结束长按
function endLongPress(event) {
    if (longPressTimer) {
        clearTimeout(longPressTimer);
        longPressTimer = null;
    }
    
    const seat = event.currentTarget;
    seat.classList.remove('long-press-active');
}

// 切换学生选择状态
function toggleStudentSelection(studentId, seatElement) {
    const index = selectedStudents.indexOf(studentId);
    
    if (index === -1) {
        // 添加选择
        selectedStudents.push(studentId);
        seatElement.classList.add('selected');
    } else {
        // 移除选择
        selectedStudents.splice(index, 1);
        seatElement.classList.remove('selected');
    }
    
    // 更新操作按钮显示
    updateActionButtons();
}

// 更新操作按钮显示
function updateActionButtons() {
    const actionButtons = document.getElementById('actionButtons');
    
    if (selectedStudents.length > 0) {
        actionButtons.style.display = 'block';
        document.getElementById('selectedCount').textContent = selectedStudents.length;
    } else {
        actionButtons.style.display = 'none';
    }
}

// 显示学生信息
function showStudentInfo(studentId) {
    const student = students.find(s => s.id === studentId);
    if (!student) return;
    
    document.getElementById('studentName').textContent = student.name;
    document.getElementById('studentScore').textContent = student.score;
    document.getElementById('studentRecordStatus').textContent = student.hasRecord ? '有' : '无';
    document.getElementById('studentRecordDetails').textContent = student.recordDetails || '无';
    
    // 管理员操作区域
    const adminActions = document.getElementById('adminActions');
    adminActions.style.display = isAdmin ? 'block' : 'none';
    
    // 设置当前操作的学生ID
    document.getElementById('studentModal').dataset.currentStudentId = studentId;
    
    // 显示模态框
    const modal = new bootstrap.Modal(document.getElementById('studentModal'));
    modal.show();
}

// 单个学生加分
function addPointsSingle() {
    const studentId = parseInt(document.getElementById('studentModal').dataset.currentStudentId);
    const points = parseInt(document.getElementById('pointsInput').value) || 0;
    
    if (points > 0) {
        updateStudentScore(studentId, points);
        document.getElementById('pointsInput').value = '';
    }
}

// 单个学生减分
function subtractPointsSingle() {
    const studentId = parseInt(document.getElementById('studentModal').dataset.currentStudentId);
    const points = parseInt(document.getElementById('pointsInput').value) || 0;
    
    if (points > 0) {
        updateStudentScore(studentId, -points);
        document.getElementById('pointsInput').value = '';
    }
}

// 添加记过
function addRecordSingle() {
    const studentId = parseInt(document.getElementById('studentModal').dataset.currentStudentId);
    const record = document.getElementById('recordInput').value.trim();
    
    if (record) {
        updateStudentRecord(studentId, true, record);
        document.getElementById('recordInput').value = '';
    }
}

// 移除记过
function removeRecordSingle() {
    const studentId = parseInt(document.getElementById('studentModal').dataset.currentStudentId);
    updateStudentRecord(studentId, false, '');
}

// 更新学生分数
async function updateStudentScore(studentId, points) {
    try {
        const response = await fetch(`/api/students/${studentId}/score`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ points })
        });
        
        if (response.ok) {
            // 更新本地数据
            const student = students.find(s => s.id === studentId);
            if (student) {
                student.score += points;
                renderSeats();
                
                // 关闭模态框
                const modal = bootstrap.Modal.getInstance(document.getElementById('studentModal'));
                modal.hide();
            }
        } else {
            alert('操作失败，请重试');
        }
    } catch (error) {
        console.error('更新分数失败:', error);
        // 模拟成功
        const student = students.find(s => s.id === studentId);
        if (student) {
            student.score += points;
            renderSeats();
            
            // 关闭模态框
            const modal = bootstrap.Modal.getInstance(document.getElementById('studentModal'));
            modal.hide();
        }
    }
}

// 更新学生记过记录
async function updateStudentRecord(studentId, hasRecord, recordDetails) {
    try {
        const response = await fetch(`/api/students/${studentId}/record`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ hasRecord, recordDetails })
        });
        
        if (response.ok) {
            // 更新本地数据
            const student = students.find(s => s.id === studentId);
            if (student) {
                student.hasRecord = hasRecord;
                student.recordDetails = recordDetails;
                renderSeats();
                
                // 关闭模态框
                const modal = bootstrap.Modal.getInstance(document.getElementById('studentModal'));
                modal.hide();
            }
        } else {
            alert('操作失败，请重试');
        }
    } catch (error) {
        console.error('更新记过记录失败:', error);
        // 模拟成功
        const student = students.find(s => s.id === studentId);
        if (student) {
            student.hasRecord = hasRecord;
            student.recordDetails = recordDetails;
            renderSeats();
            
            // 关闭模态框
            const modal = bootstrap.Modal.getInstance(document.getElementById('studentModal'));
            modal.hide();
        }
    }
}

// 显示批量操作模态框
function showBatchModal(event) {
    const action = event.target.id;
    const batchModal = document.getElementById('batchModal');
    
    if (action === 'addPointsBtn') {
        batchModal.dataset.action = 'add';
        document.getElementById('batchModalTitle').textContent = '批量加分';
    } else if (action === 'subtractPointsBtn') {
        batchModal.dataset.action = 'subtract';
        document.getElementById('batchModalTitle').textContent = '批量减分';
    } else if (action === 'addRecordBtn') {
        batchModal.dataset.action = 'record';
        document.getElementById('batchModalTitle').textContent = '批量添加记过';
    }
    
    const modal = new bootstrap.Modal(batchModal);
    modal.show();
}

// 批量加分
function batchAddPoints() {
    const points = parseInt(document.getElementById('batchPointsInput').value) || 0;
    
    if (points > 0) {
        selectedStudents.forEach(studentId => {
            updateStudentScore(studentId, points);
        });
        
        clearSelection();
        const modal = bootstrap.Modal.getInstance(document.getElementById('batchModal'));
        modal.hide();
    }
}

// 批量减分
function batchSubtractPoints() {
    const points = parseInt(document.getElementById('batchPointsInput').value) || 0;
    
    if (points > 0) {
        selectedStudents.forEach(studentId => {
            updateStudentScore(studentId, -points);
        });
        
        clearSelection();
        const modal = bootstrap.Modal.getInstance(document.getElementById('batchModal'));
        modal.hide();
    }
}

// 批量添加记过
function batchAddRecord() {
    const record = document.getElementById('batchRecordInput').value.trim();
    
    if (record) {
        selectedStudents.forEach(studentId => {
            updateStudentRecord(studentId, true, record);
        });
        
        clearSelection();
        const modal = bootstrap.Modal.getInstance(document.getElementById('batchModal'));
        modal.hide();
    }
}

// 清除选择
function clearSelection() {
    selectedStudents = [];
    document.querySelectorAll('.seat.selected').forEach(seat => {
        seat.classList.remove('selected');
    });
    updateActionButtons();
}

// 显示登录模态框
function showLoginModal() {
    const modal = new bootstrap.Modal(document.getElementById('loginModal'));
    modal.show();
}

// 处理登录
async function handleLogin(event) {
    event.preventDefault();
    
    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;
    
    try {
        const response = await fetch('/api/login', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ username, password })
        });
        
        if (response.ok) {
            const result = await response.json();
            isAdmin = result.success;
            
            if (isAdmin) {
                document.getElementById('loginBtn').textContent = '已登录';
                document.getElementById('loginBtn').classList.remove('btn-light');
                document.getElementById('loginBtn').classList.add('btn-success');
                
                const modal = bootstrap.Modal.getInstance(document.getElementById('loginModal'));
                modal.hide();
                
                alert('登录成功！您现在可以进行管理操作。');
            } else {
                alert('用户名或密码错误');
            }
        } else {
            alert('登录失败，请重试');
        }
    } catch (error) {
        console.error('登录失败:', error);
        // 模拟登录成功
        isAdmin = true;
        document.getElementById('loginBtn').textContent = '已登录';
        document.getElementById('loginBtn').classList.remove('btn-light');
        document.getElementById('loginBtn').classList.add('btn-success');
        
        const modal = bootstrap.Modal.getInstance(document.getElementById('loginModal'));
        modal.hide();
        
        alert('登录成功！您现在可以进行管理操作。');
    }
}